# Copyright (C) Jan 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --


#######################################################
#
# Parser.py
# Python implementation of the Class Parser
# Generated by Enterprise Architect
# Created on:      19-Dec-2019 3:18:38 PM
# Original author: talve
#
#######################################################
import os
import math
import re
import struct

from resourceparse_lib.utils import constants as cs
from resourceparse_lib.resource_data.RawData import RawData
from resourceparse_lib.resource_data.DataPrinter import DataPrinter
from resourceparse_lib.utils.Exceptions import ResourceParseException
from resourceparse_lib.parsers.RawParser import RawParser


class ResourceParseManager:
    """This class responsible for parsing the segments according the given raw data
    and adb segments by organize the given inputs and print them after the parsing
    posses.
    """
    def __init__(self, manager_args, parser_args, segments):
        """This method initialize the class members
        """
        self._verbosity = manager_args.verbose
        self._out_file = manager_args.out
        self._out_dir = manager_args.out_dir
        self._printer = DataPrinter(self._verbosity, self._out_file, self._out_dir)
        self._warning_counter = 0
        self._error_counter = 0
        self._notice_counter = 0

        try:
            self._dumped_segment_db = segments if segments else self._retrieve_dumped_segment_db(manager_args.dump_file, manager_args.resource_parser)
        except ResourceParseException as rpe:
            raise ResourceParseException("{0}\nFail to generate segment db from raw data.".format(rpe))

        try:
            parser_args.manager = self
            self._parser = manager_args.resource_parser(parser_args)
        except ResourceParseException as rpe:
            raise ResourceParseException("{0}\nFailed to parse with parser - {1}.".format(rpe, self._parser.__name__))

        self._parser.validate()

    def _get_next_warning_counter(self):
        """This method return get the next free index to help generate
         the a uniq key for the warning msg.
        """
        returned_value = self._warning_counter
        self._warning_counter += 1
        return returned_value

    def _get_next_error_counter(self):
        """This method return get the next free index to help generate
        the a uniq key for the error msg.
        """
        returned_value = self._error_counter
        self._error_counter += 1
        return returned_value

    def _get_next_notice_counter(self):
        """This method return get the next free index to help generate
        the a uniq key for the notice msg.
        """
        returned_value = self._notice_counter
        self._notice_counter += 1
        return returned_value

    def parse(self):
        """This method parse the segment according the the dumped input file and the
        current configurations.
        """
        # parse all segments
        seg_parsed_counter = 0
        try:
            for seg in self._dumped_segment_db:
                seg_parsed_counter += 1
                self._parser.parse_segment(seg)
        except ResourceParseException as rpe:
            raise ResourceParseException("{0}\nFail to parse segments, failure occur at segment number {1}.".format(rpe, seg_parsed_counter))

        # print all segments
        self._printer.print_parsed_segment(self._dumped_segment_db, self._parser.get_title(), self._parser.get_segment_separator())

    def retrieve_adb_version_from_info_segment(self):
        """This method locate the info segment and if found, return the
        fw_version at the format XX.XX.XXXX, otherwise return empty string.
        """
        for seg in self._dumped_segment_db:
            if seg.get_type() == cs.RESOURCE_DUMP_SEGMENT_TYPE_INFO:
                return seg.get_version()
        return ""

    def get_num_dumped_segments(self):
        return len(self._dumped_segment_db)

    def _retrieve_dumped_segment_db(self, dumped_file_path, parser_type):
        """This method get the segment list generated from the dumped input file by calling
        the RawData class.
        """
        return RawData(dumped_file_path).to_segments(parser_type is not RawParser)
